module database	// Small database program to manipulate a simple database

//	The program has been written in Clean 1.3.1 and uses the Clean Standard Object I/O library 1.1

import StdEnv, StdIO
import listextensions

::	Record
	:==	[ String ]							// [Content]
::	Descriptor
	:== [ String ]							// [Fieldname]
::	DataBase
	=	{	records		:: [Record]			// All records
		,	descriptor	:: Descriptor		// All fieldnames
		,	selection	:: Int				// Indicating current record selected
		,	query		:: Record			// Record to look for
		,	name		:: String			// Name of database
		,	editinfoid	:: Id				// Id of info about use of edit dialog (query or record)
		,	edittextids	:: [Id]				// Ids of the edit text fields of the edit dialog
		,	fw			:: Int				// Max width of field contents
		,	dw			:: Int				// Max width of descriptor fields
		,	dbfont		:: InfoFont			// The database font information
		,	dffont		:: InfoFont			// The dialog   font information
		}
::	InfoFont
	=	{	font		:: Font				// The font which is used
		,	width		:: Int				// Its widest character
		,	height		:: Int				// Its line height
		}
::	NoState
	=	NoState

// fopen for use with accFiles
fopen2 fileName mode files
	:==	((ok, file), files2)
	where
		(ok, file, files2)
			=	fopen fileName mode files

MinDbDomainSize :== {w=100,h=1}				// Minimal size of recordwindow
CharsInInputBox :== 20						// Input width (number of characters)

Replace			:== True					// Replace current selection when adding new record
Separator		:==	": "					// Separates field names and contents
CurrentRecordNr	:==	"Current Record Nr: "	// The text field that displays the current record nr

Start :: *World -> *World
Start world
	#	((_,dbFont),    world)	= accScreenPicture (openFont NonProportionalFontDef) world
	#	(dfFont,        world)	= accScreenPicture openDialogFont world
	#	(dbInfo,        world)	= accScreenPicture (getInfoFont dbFont) world
	#	(dfInfo,        world)	= accScreenPicture (getInfoFont dfFont) world
	#	(recordWindowId,world)	= openId world				// Id of window that shows the records
	#	(edDialogId,    world)	= openId world				// Id of window that edits a record
	#	(fieldDialogId, world)	= openId world				// Id of window that edits the set up
	#	(editInfoId,    world)	= openId world				// Id of text that shows current record nr
		initState				= {	records		= []
								  ,	descriptor	= []
								  ,	selection	= 0
								  ,	query		= []
								  ,	name		= ""
								  ,	editinfoid	= editInfoId
								  ,	edittextids	= []
								  ,	fw			= 0
								  ,	dw			= 0
								  ,	dbfont		= dbInfo
								  ,	dffont		= dfInfo
								  }
	=	startIO initState NoState (initIO (recordWindowId,edDialogId,fieldDialogId)) [] world
where
	getInfoFont :: Font *Picture -> (InfoFont,*Picture)
	getInfoFont font picture
		#	(metrics,picture)	= getFontMetrics font picture
		=	({ font=font,width=metrics.fMaxWidth,height=fontLineHeight metrics },picture)

initIO :: (Id,Id,Id) -> ProcessInit (PSt DataBase .p)
initIO ids=:(recordWindowId,edDialogId,fieldDialogId)
	=	[snd o openMenu undef menu, ReadDataBase, ShowRecords, ShowEditDialog ]
where
	menu	= Menu "Commands"
				(	MenuItem "Show Records"		[MenuShortKey 'r', MenuFunction (noLS ShowRecords)]
				:+:	MenuItem "Edit..."			[MenuShortKey 'e', MenuFunction (noLS ShowEditDialog)]
				:+:	MenuItem "Change Set Up..."	[MenuShortKey 'u', MenuFunction (noLS ShowFieldDialog)]
				:+:	MenuItem "Read new..."		[MenuShortKey 'o', MenuFunction (noLS ReadNew)]
				:+:	MenuItem "Save As..."		[MenuShortKey 's', MenuFunction (noLS SaveRecords)]
				:+:	MenuItem "Print"			[MenuShortKey 'p', MenuFunction (noLS Print)]
				:+:	MenuSeparator				[]
				:+:	MenuItem "Quit"				[MenuShortKey 'q', MenuFunction (noLS closeProcess)]
				)	[]
	
//	The callback and initialisation functions of the menu:
	
	ReadNew :: (PSt DataBase .p) -> PSt DataBase .p
	ReadNew database
		=	seq (map closeWindow [recordWindowId,edDialogId,fieldDialogId] ++ tl (initIO ids)) database
	
	ReadDataBase :: (PSt DataBase .p) -> PSt DataBase .p
	ReadDataBase database=:{ls={dbfont={font}}}
		#	(maybe_dbname,database)	= selectInputFile database
		|	isNothing maybe_dbname
			=	database
		#	dbname					= fromJust maybe_dbname
		#	((open,dbfile),database)= accFiles (fopen2 dbname FReadText) database
		|	not open
			=	appPIO beep database
		#	(descr,dbfile)			= FReadDescr dbfile
		#	(recs, dbfile)			= FReadRecords (length descr+1) dbfile	// lines = length descr + empty line
		#	(close,database)		= accFiles (fclose dbfile) database
		|	not close
			=	appPIO beep database
		#	(fw,database)			= accPIO (accScreenPicture (MaxWidth font (flatten recs))) database
		#	(dw,database)			= accPIO (accScreenPicture (MaxWidth font descr)) database
		|	otherwise
		=	appPLoc (\db->{db & records		= recs
							  ,	descriptor	= descr
							  ,	query		= repeatn (length descr) ""
							  ,	selection	= 0
							  ,	name		= dbname
							  ,	fw			= fw
							  ,	dw			= dw
						   }) database
	where
		FReadDescr file
			#	(nroffields,file)	= FReadStrippedLine file
			#	(descr,file)		= seqList (repeatn (toInt nroffields) FReadStrippedLine) file
			=	(descr,file)
		
		FReadRecords nroflines file
			|	sfend file
				=	([], file)
			#	([_:record],file)	= seqList (repeatn nroflines FReadStrippedLine) file
			#	(records,   file)	= FReadRecords nroflines file
			|	otherwise
				=	([record : records], file)
		
		FReadStrippedLine file
			#	(line,file)	= freadline file
			=	(line%(0,size line - 2),file)		// strip "\n"
	
	ShowRecords :: (PSt DataBase .p) -> PSt DataBase .p
	ShowRecords database=:{ls=state=:{records,descriptor,dw,name,dbfont}}
		#	(domain,database)	= accPIO (accScreenPicture (DbViewDomain state 0 (max (length records) 1))) database
		#	(_,database)		= openWindow undef (window domain) database
		=	database
	where
		window domain
			=	Window namewithoutdirectories NilLS
					[	WindowId			recordWindowId
					,	WindowPos			(LeftTop,{vx=5,vy=5})
					,	WindowMouse			(const True) Able (noLS1 MouseSelectItem)
					,	WindowHScroll		(hscroll dbfont.width)
					,	WindowVScroll		(vscroll dbfont.height)
					,	WindowViewDomain	domain
					,	WindowMinimumSize	MinDbDomainSize
					,	WindowLook			(RecordWindowLook state)
					,	WindowResize
					]
		namewithoutdirectories
			=	toString (last (splitby dirseparator (fromString name)))
		
		hscroll dh viewframe {sliderThumb} move
			=	case move of
					SliderIncSmall -> sliderThumb+dh
					SliderDecSmall -> sliderThumb-dh
					SliderIncLarge -> sliderThumb+(rectangleSize viewframe).w*9/10
					SliderDecLarge -> sliderThumb-(rectangleSize viewframe).w*9/10
					SliderThumb x  -> x
		vscroll dv viewframe {sliderThumb} move
			=	case move of
					SliderIncSmall -> sliderThumb+dv
					SliderDecSmall -> sliderThumb-dv
					SliderIncLarge -> sliderThumb+(rectangleSize viewframe).h*9/10
					SliderDecLarge -> sliderThumb-(rectangleSize viewframe).h*9/10
					SliderThumb y  -> y
	
	ShowEditDialog :: (PSt DataBase .p) -> PSt DataBase .p
	ShowEditDialog database=:{ls=state=:{descriptor=descr,records=recs,selection,dffont,editinfoid}}
		#	(ids,database)	= accPIO (openIds nr_of_ids) database
		#	database		= {database & ls={state & edittextids=tl ids}}
		#	(_,database)	= openDialog undef (editDialog ids) database
		#	database		= SetTextFields infostring descr (if (isEmpty recs) [] (recs!!selection)) database
		=	database
	where
		nr_descrs			= length descr
		nr_of_ids			= nr_descrs + 1		// Generate Ids for descriptor fields and default button ("Add")
		inputboxwidth		= CharsInInputBox*dffont.width
		infostring			= CurrentRecordNr+++toString selection
		editDialog ids
			=	Dialog "Edit Record" 
					(	TextControl "" [ControlId editinfoid,ControlSize {w=inputboxwidth,h=dffont.height}	]
					:+:	CompoundControl
					(	ListLS
					[	TextControl field [ControlPos (Left,zero)] \\ field <- descr	]
					)	[ControlPos (Left,zero)]
					:+:	CompoundControl
					(	ListLS
					[	EditControl "" inputboxwidth 1 [ControlId (ids!!i),ControlPos (Left,zero)] \\ i<-[1..nr_descrs]	]
					)	[]
					:+:	ButtonControl "DisplQ"		[ControlFunction (noLS DisplQuery),ControlPos (Left,zero)]
					:+:	ButtonControl "SetQ"		[ControlFunction (noLS SetQuery)]
					:+:	ButtonControl "SearchQ" 	[ControlFunction (noLS Search)]
					:+:	ButtonControl "SelectAllQ"	[ControlFunction (noLS SelectAll)]
					:+:	ButtonControl "Replace"		[ControlFunction (noLS (AddRecord Replace)),ControlPos (Left,zero)]
					:+:	ButtonControl "Delete"		[ControlFunction (noLS DeleteRecord)]
					:+:	ButtonControl "Add"			[ControlFunction (noLS (AddRecord (not Replace))),ControlId addId]
					:+:	ButtonControl "Sort"		[ControlFunction (noLS Sort)]
					)
					[	WindowId	edDialogId
					,	WindowOk	addId
					]
		where
			addId			= ids!!0
	
	ShowFieldDialog :: (PSt DataBase .p) -> PSt DataBase .p
	ShowFieldDialog database=:{ls=state=:{descriptor=d,dffont}}
		|	isEmpty d
			#	inputboxwidth	= CharsInInputBox*dffont.width
			=	inputdialog "Give first field" inputboxwidth (\input->FieldChangeIO (add (-1) input)) database
		|	otherwise
			#	database		= closeWindow edDialogId database
			#	(ids,database)	= accPIO (openIds 2) database
			#	(_,database)	= openDialog undef (fielddialog ids) database
			=	database
	where
		fielddialog ids
			=	Dialog "Change Set Up" 
					(	TextControl "Select Field..." []
					:+:	RadioControl (radioitems d) (Columns 1) 1 [ControlId selectId]
					:+:	CompoundControl
					(	ButtonControl "Delete"	[	ControlPos      (Left,zero)
												,	ControlFunction (noLS (DeleteField fieldDialogId getselectedfield))
												]
					:+:	ButtonControl "Rename"	[	ControlPos      (Left,zero)
												,	ControlFunction (noLS (RenameField fieldDialogId getselectedfield))
												]
					)	[ControlPos (Left,zero)]
					:+:	CompoundControl
					(	ButtonControl "Move"	[	ControlPos      (Left,zero)
												,	ControlFunction (noLS (MoveField fieldDialogId getselectedfield))
												]
					:+:	ButtonControl "Add New"	[	ControlPos      (Left,zero)
												,	ControlId       addId
												,	ControlFunction (noLS (AddField fieldDialogId getselectedfield))
												]
					)	[]
					)
					[	WindowId	fieldDialogId
					,	WindowOk	addId
					]
		where
			getselectedfield dialoginfo
				=	fromJust (snd (hd (getRadioControlSelection [selectId] dialoginfo))) - 1
			[addId,selectId:_]
				=	ids
	
	SaveRecords :: (PSt DataBase .p) -> PSt DataBase .p
	SaveRecords database=:{ls=state=:{name,descriptor,records}}
		#	(maybe_dbname,database)	= selectOutputFile "Save As: " name database
		|	isNothing maybe_dbname
			=	database
		#	dbname					= fromJust maybe_dbname
		#	((open,dbfile),database)= accFiles (fopen2 dbname FWriteText) database
		|	not open
			=	appPIO beep database
		#	(close,database)		= accFiles (fclose (seq (writedescriptor++writerecords) dbfile)) database
		|	close
			=	database
		|	otherwise
			=	appPIO beep database
	where
		writedescriptor		= [fwritei (length descriptor), FWriteRecord descriptor]
		writerecords		= [FWriteRecord rec \\ rec <- records]
		FWriteRecord rec	= fwrites (foldl (+++) "\n" (map (\field -> field +++ "\n") rec))
	
	// Field set up changes
	
	FieldChangeIO :: (IdFun (PSt DataBase .p)) (PSt DataBase .p) -> PSt DataBase .p
	FieldChangeIO changefun database
		#	database	= changefun database
		#	database	= seq (map closeWindow [fieldDialogId,edDialogId]) database
		#	database	= UpdateDbDomain database
		=	database
	
	AddField :: Id (WState -> Int) (PSt DataBase .p) -> PSt DataBase .p
	AddField windowid getfield database=:{ls=state=:{dffont}}
		#	(maybe_wstate,database)	= accPIO (getWindow windowid) database
		|	isNothing maybe_wstate
			=	appPIO beep database
		|	otherwise
			#	wstate					= fromJust maybe_wstate
				fieldname				= getfield wstate
				infotext				= "Add after '"+++state.descriptor!!fieldname+++"' new field"
				inputboxwidth			= CharsInInputBox*dffont.width
			=	inputdialog infotext inputboxwidth (\input->FieldChangeIO (add fieldname input)) database
	
	RenameField :: Id (WState -> Int) (PSt DataBase .p) -> PSt DataBase .p
	RenameField windowid getfield database=:{ls=state=:{dffont}}
		#	(maybe_wstate,database)	= accPIO (getWindow windowid) database
		|	isNothing maybe_wstate
			=	appPIO beep database
		|	otherwise
			#	wstate					= fromJust maybe_wstate
				fieldtorename			= getfield wstate
				infotext				= "Rename '"+++state.descriptor!!fieldtorename+++"' to"
				inputboxwidth			= CharsInInputBox*dffont.width
			=	inputdialog infotext inputboxwidth (\input->FieldChangeIO (rename fieldtorename input)) database
	
	MoveField :: Id (WState -> Int) (PSt DataBase .p) -> PSt DataBase .p
	MoveField windowid getfield database=:{ls={descriptor=d}}
		#	(maybe_wstate,database)	= accPIO (getWindow windowid) database
		|	isNothing maybe_wstate
			=	appPIO beep database
		|	otherwise
			#	wstate				= fromJust maybe_wstate
				fieldtomove			= getfield wstate
				(ids,database)		= accPIO (openIds 2) database
			=	snd (openDialog undef (movedialog ids fieldtomove) database)
	where
		movedialog ids fieldtomove
			=	Dialog "Move Field"
					(	TextControl   ("Move '" +++ d!!fieldtomove +++ "' before: ") []
					:+:	RadioControl  (radioitems (d++[""])) (Rows (length d+1)) 1
												[ControlId selectId,    ControlPos      (Left,zero)]
					:+:	ButtonControl Cancel	[ControlPos (Left,zero),ControlFunction (noLS cancel)]
					:+:	ButtonControl "Move"	[ControlId okId,        ControlFunction (noLS (ok (move fieldtomove)))]
					)
					[	WindowOk okId	]
		where
			[okId, selectId:_]
				=	ids
			
			ok mvf pState
				#	(maybe_id,pState)		= accPIO getActiveWindow pState
				|	isNothing maybe_id
					=	pState
				|	otherwise
					#	id						= fromJust maybe_id
					#	(Just dialoginfo,pState)= accPIO (getWindow id) pState
						destinationfield		= fromJust (snd (hd (getRadioControlSelection [selectId] dialoginfo)))-1
					=	FieldChangeIO (mvf destinationfield) (closeWindow id pState)
	
	DeleteField :: Id (WState -> Int) (PSt DataBase .p) -> PSt DataBase .p
	DeleteField windowid getfield database
		#	(maybe_wstate,database)	= accPIO (getWindow windowid) database
		|	isNothing maybe_wstate
			=	appPIO beep database
		|	otherwise
			#	wstate				= fromJust maybe_wstate
			=	warn ["Are you sure?"] (FieldChangeIO (delete (getfield wstate))) database
	
	//	Handling the edit dialog
	
	DisplQuery :: (PSt DataBase .p) -> PSt DataBase .p
	DisplQuery database=:{ls={descriptor,query}}
		=	SetTextFields "Query :" descriptor query database
	
	SetQuery :: (PSt DataBase .p) -> PSt DataBase .p
	SetQuery database=:{ls=state}
		#	(nquery,database)	= GetTextFields state.descriptor database
		#	database			= {database & ls={state & query = nquery}}
		=	database
	
	Search :: (PSt DataBase .p) -> PSt DataBase .p
	Search database=:{ls=state=:{records,query,selection=sel,edittextids},io}
		|	isEmpty found
			=	appPIO beep database
		|	otherwise
			#	database	= ChangeSelection edittextids sel nsel {database & ls = {state & selection=nsel}}
			#	database	= MakeSelectionVisible database
			=	database
	where
		nsel	= hd found
		found	= [i \\ e <- el ++ bl & i <- [sel+1 .. length records - 1] ++ [0..] | QueryRecord query e]
		(bl,el)	= splitAt (sel+1) records
	
	QueryRecord :: Record Record -> Bool
	QueryRecord query e
		=	and [ EqPref qf f \\ f <- e & qf <- query ]
	where
		EqPref pref name
			|	size_pref > size_name
				=	False
			|	otherwise
				=	pref == name%(0,size_pref - 1) || EqPref pref (name%(1,size_name - 1))
		where
			size_pref	= size pref
			size_name	= size name
	
	SelectAll :: (PSt DataBase .p) -> PSt DataBase .p
	SelectAll database=:{ls=state=:{records,query,selection,descriptor,dbfont,edittextids}}
		#	recs				= filter (QueryRecord query) records
		|	isEmpty recs
			=	appPIO beep database
		|	otherwise
			#	(fw,database)	= accPIO (accScreenPicture (MaxWidth dbfont.font (flatten recs))) database
			#	database		= appPIO (setWindowTitle recordWindowId "Select") database
				state			= {state & selection=0,records=recs,name="Select",fw=fw}
			#	database		= {database & ls=state}
			#	database		= ChangeSelection edittextids selection 0 database
			#	database		= UpdateDbDomain database
			=	database
	
	MakeSelectionVisible :: (PSt DataBase .p) -> PSt DataBase .p
	MakeSelectionVisible database=:{ls=state=:{records,selection,descriptor,dbfont},io}
		|	isEmpty records
			=	database
		#	(viewframe,io)	= getWindowViewFrame recordWindowId io
			visibletop		= viewframe.corner1.y
			visiblebot		= viewframe.corner2.y
		|	selthumb >= visibletop && selthumb < visiblebot		// selection visible
			=	{database & io=io}
		|	otherwise											// selection invisible
			=	{database & io=moveWindowViewFrame recordWindowId {vx=0,vy=selthumb-visibletop} io}
	where
		selthumb			= toPicCo dbfont descriptor selection
	
	DeleteRecord :: (PSt DataBase .p) -> PSt DataBase .p
	DeleteRecord database=:{ls=state=:{records=oldrecs,selection=index,descriptor,fw,dbfont}}
		|	isEmpty oldrecs
			=	appPIO beep database
		#	(indexwidth,database)	= accPIO (accScreenPicture (MaxWidth dbfont.font (oldrecs!!index))) database
		|	fw <> indexwidth
			=	UpdateDbDomain {database & ls={state & records = newrecs, selection = nindex}}
		#	(fieldwidth,database)	= accPIO (accScreenPicture (MaxWidth dbfont.font (flatten newrecs))) database
		|	otherwise
			=	UpdateDbDomain {database & ls={state & records = newrecs, selection = nindex, fw = fieldwidth}}
	where
		newrecs	= removeAt index oldrecs
		nindex	= if (isEmpty newrecs) 0 (index mod length newrecs) 
	
	AddRecord :: Bool (PSt DataBase .p) -> PSt DataBase .p
	AddRecord replace database=:{ls=state=:{descriptor,selection,records=recs,fw,dbfont},io}
		|	isEmpty recs && replace
			=	appPIO beep database
		|	otherwise
			#	(newrec,database)			= GetTextFields descriptor database
				(index,newrecs)				= insertindex (<=) newrec (if replace (removeAt selection recs) recs)
			#	(selectionwidth,database)	= accPIO (accScreenPicture (MaxWidth dbfont.font (recs!!selection))) database
			#	(newrecswidth,  database)	= accPIO (accScreenPicture (MaxWidth dbfont.font (flatten newrecs))) database
			#	(newrecwidth,   database)	= accPIO (accScreenPicture (MaxWidth dbfont.font newrec)) database
				recalc						= replace && selectionwidth < fw
				fieldwidth					= if recalc newrecswidth (max newrecwidth fw)
				nstate						= {state & records=newrecs,selection=index,fw=fieldwidth}
			=	UpdateDbDomain {database & ls=nstate}
	
	Sort :: (PSt DataBase .p) -> PSt DataBase .p
	Sort database=:{ls=state=:{records=recs}}
		=	UpdateDbDomain {database & ls={state & records = sort recs}}
	
	GetTextFields :: Descriptor (PSt DataBase .p) -> (Record,PSt DataBase .p)
	GetTextFields descr database=:{ls={edittextids}}
		#	(Just dialog,database) = accPIO (getWindow edDialogId) database
		=	(map (\(_,Just text)->text) (getControlTexts edittextids dialog),database)
	
	SetTextFields :: String Descriptor Record (PSt DataBase .p) -> PSt DataBase .p
	SetTextFields s d rec database=:{ls={editinfoid,edittextids}}
		=	appPIO (setWindow edDialogId [setControlTexts (zip2 [editinfoid:edittextids] [s:rec])]) database
	
	//	Handling mouse clicks in database window
	
	MouseSelectItem	:: MouseState (PSt DataBase .p) -> PSt DataBase .p
	MouseSelectItem (MouseDown {y} _ _) database=:{ls=state=:{records,descriptor,selection,dbfont,edittextids},io}
		|	isEmpty records
			=	database
		|	otherwise
			#	(Just viewDomain,database)	= accPIO (getWindowViewDomain recordWindowId) database
			#	index						= toRecCo dbfont descriptor (min y (viewDomain.corner2.y-1))
			#	nstate						= {state & selection=index}
			=	ChangeSelection edittextids selection index {database & ls=nstate}
	MouseSelectItem _ database
		=	database
	
	//	Update the whole window in case the ViewDomain has changed
	
	UpdateDbDomain :: (PSt DataBase .p) -> PSt DataBase .p
	UpdateDbDomain database=:{ls=state}
		#	(viewdomain,database)
						= accPIO (accScreenPicture (DbViewDomain state 0 (max (length state.records) 1))) database
		#	database	= appPIO (setWindowLook recordWindowId True (RecordWindowLook state)) database
		#	database	= appPIO (setWindowViewDomain recordWindowId viewdomain) database
		#	database	= MakeSelectionVisible database
		=	database

	ChangeSelection :: [Id] Int Int (PSt DataBase .p) -> PSt DataBase .p
	ChangeSelection edittextids old new database=:{ls=state=:{descriptor=descr,records},io}
		#	io		= drawInWindow recordWindowId [HiliteSelection state old, HiliteSelection state new] io
		#	io		= setWindowLook recordWindowId False (RecordWindowLook state) io
		#	database= SetTextFields infostring descr (records!!new) {database & io=io}
		=	database
	where
		infostring	= CurrentRecordNr+++toString new

//	Functions that change the content of particular fields

add afterfield fieldname database=:{ls=state=:{records=rs,descriptor=d,query=q,dw,dbfont}}
	#	(widths,database)	= accPIO (accScreenPicture (MaxWidth dbfont.font [fieldname])) database
		descrwidth			= max widths dw
	=	{database & ls={state & records		= map (ins "") rs
							  ,	descriptor	= ins fieldname d
							  ,	query		= ins "" q
							  ,	dw			= descrwidth
					   }
		}
where
	ins x ys   = insertAt (afterfield+1) x ys

rename selectedfield newfieldname database=:{ls=state=:{descriptor=d,dbfont}}
	#	(width,database)	= accPIO (accScreenPicture (MaxWidth dbfont.font newdescr)) database
	=	{database & ls={state & descriptor	= newdescr
							  ,	dw			= width
					   }
		}
where
	newdescr = updateAt selectedfield newfieldname d
	
move sf df database=:{ls=state=:{records=rs,descriptor=d,query=q}}
	=	{database & ls={state & records		= map (moveinlist sf df) rs
							  ,	descriptor	= moveinlist sf df d
							  ,	query		= moveinlist sf df q
					   }
		}

delete i database=:{ls=state=:{records=rs,descriptor=d,query=q,dbfont}}
	#	(width,database) = accPIO (accScreenPicture (MaxWidth dbfont.font newdescr)) database
		(nfw,  database) = accPIO (accScreenPicture (MaxWidth dbfont.font (flatten newrs))) database
	=	{database & ls={state & records		= newrs
							  ,	descriptor	= newdescr
							  ,	query		= removeAt i q
							  ,	dw			= width
							  ,	fw			= nfw
					   }
		}
where
	newrs    = map (removeAt i) rs
	newdescr = removeAt i d

//	Drawing utilities

DbViewDomain :: DataBase Int Int *Picture -> (ViewDomain,*Picture)
DbViewDomain {descriptor=d,records,dw,fw,dbfont} fr to picture
	#	(separatorwidth,picture)
					= MaxWidth dbfont.font [Separator] picture
		whiteMargin	= dbfont.width
		viewdomain	= {	corner1 = {x = ~whiteMargin                          ,y = toPicCo dbfont d fr }
					  ,	corner2 = {x = dw + separatorwidth + fw + whiteMargin,y = toPicCo dbfont d to }
					  }
	|	rectangleSize viewdomain < MinDbDomainSize
		=	({ corner1={x= ~whiteMargin,y = 0},corner2={x= ~whiteMargin+MinDbDomainSize.w,y= MinDbDomainSize.h}},picture)
	|	otherwise
		=	(viewdomain,picture)

RecordWindowLook :: DataBase SelectState UpdateState -> *Picture -> *Picture
RecordWindowLook state=:{records=recs,descriptor=descr,selection,dbfont} _ {updArea=domains} 
	=	seq [setPenFont dbfont.font : flatten (map Update domains) ++ [HiliteSelection state selection]]
where 
	Update domain=:{corner1={y=top},corner2={y=bottom}}
		=	[	setPenColour White
			,	fill domain
			,	setPenColour Black
			:	if (isEmpty recs)
					[]
					[setPenPos {x=0,y=topofvisiblerecs} : map (DrawRec descr) (recs%(toprec,botrec))]
			]
	where
		topofvisiblerecs= toPicCo dbfont descr toprec
		toprec			= toRecCo dbfont descr top
		botrec			= toRecCo dbfont descr (bottom-1)
	
	DrawRec descr rec
		=	seq [drawLine "" : [drawLine (d +++ Separator +++ f) \\ d<-normwidth descr & f<-rec]]
	where
		normwidth descr = [f +++ toString (spaces ((maxList (map (size ) descr)) - size f)) \\ f <- descr]
		drawLine s picture
			#	(curPenPos, picture)= getPenPos picture
			#	picture				= draw s picture
			#	picture				= setPenPos {curPenPos & y=curPenPos.y+dbfont.height} picture
			=	picture

HiliteSelection :: DataBase Int *Picture -> *Picture
HiliteSelection s i pict
	#	(selection,pict)	= DbViewDomain s i (i+1) pict
	=	hilite selection pict

//	Switching between picture coordinates and indices in the list of records ('record coordinates')

toPicCo:: InfoFont Descriptor Int -> Int
toPicCo dbfont descr n = n * (inc (length descr) * dbfont.height)

toRecCo:: InfoFont Descriptor Int -> Int
toRecCo dbfont descr n = n / (inc (length descr) * dbfont.height)

// Various useful functions

instance < Size where
	(<) :: !Size !Size -> Bool
	(<) {w=w1,h=h1} {w=w2,h=h2} = w1<w2 && h1<h2

radioitems titles
	=	[(t,id) \\ t <- titles]

MaxWidth :: Font ![String] !*Picture -> (!Int,!*Picture)
MaxWidth font [] picture
	=	(0,picture)
MaxWidth font list picture
	#	(widths,picture)	= getFontStringWidths font list picture
	=	(maxList widths,picture)

// functions that should be library functions

Cancel		:==	"Cancel"
OK			:==	"OK"

inputdialog name width fun pState
	#	(ids,pState)	= accPIO (openIds 2) pState
	#	(_,pState)		= openDialog undef (dialogdef ids) pState
	=	pState
where
	dialogdef ids
		=	Dialog name
				(	TextControl  (name+++": ")	[]
				:+: EditControl  "" width 1		[ControlId inputId]
				:+:	ButtonControl Cancel		[ControlPos (BelowPrev,zero),ControlFunction (noLS cancel)]
				:+:	ButtonControl OK			[ControlId okId,             ControlFunction (noLS (ok fun))]
				)
				[	WindowOk	okId
				]
	where
		[inputId,okId:_]
			=	ids
		
		ok fun pState
			#	(Just id,    pState)	= accPIO getActiveWindow pState
			#	(Just dialog,pState)	= accPIO (getWindow id) pState
				input					= fromJust (snd (hd (getControlTexts [inputId] dialog)))
			=	fun input (closeWindow id pState)

warn info fun pState
	#	(id,pState)	= accPIO openId pState
	#	(_,pState)	= openModalDialog undef (warningdef id) pState
	=	pState
where
	warningdef okId
		=	Dialog "Warning!"
				(	ListLS
				[	TextControl line		[ControlPos (Left,zero)] \\ line <- info	]
				:+:	ButtonControl Cancel	[ControlFunction (noLS cancel),ControlPos (Center,zero)]
				:+:	ButtonControl OK		[ControlFunction (noLS (fun o cancel)),ControlId okId]
				)
				[	WindowOk okId	]

cancel pState
	#	(Just id,pState)	= accPIO getActiveWindow pState
	=	closeWindow id pState

Print :: (PSt DataBase .p) -> PSt DataBase .p
Print pSt=:{ls={records,descriptor,name,dw,dbfont}}
	#!	(defaulPS, pSt)	= accPIO defaultPrintSetup pSt
	= snd (print True True pages defaulPS pSt)
  where
	pages {printSetup, jobInfo={range=(first,last), copies}} picture1
		= (map print_page groups_printed, picture2)
	  where
		{page={w=width,h=height}}	= getPageDimensions printSetup True
		({fAscent, fDescent, fLeading}, picture2) = getFontMetrics dbfont.font picture1
		line_height = dbfont.height
		record_height = (1+(length descriptor))*line_height
		table_height = height - 2*line_height
		nr_records_per_page = table_height/record_height		// fa: first approximation
		groups = group_by nr_records_per_page records
		groups_printed = flatten (repeatn copies (groups % (first-1,last-1)))
	
		print_page record_group picture
			# line_pos_y = 3*line_height/2
			  picture = seq [	setPenFont dbfont.font,
								drawAt {x=0, y=fAscent+fLeading} name,
								drawLine { x=0, y=line_pos_y} { x=width, y=line_pos_y}
							] picture
			= seq [	print_record (n*record_height+2*line_height) descriptor field_values 
					\\ n<-[0..] & field_values<-record_group ]
				  picture
	
		print_record y field_names field_values picture
			= seq [ draw_row s1 s2 n \\ s1<-field_names & s2<-field_values & n<-[1..]] picture
		  where
//			draw_row (AttDesc s1 STRING) (AS s2) n picture
			draw_row s1 s2 n picture
				# baseline = y+n*line_height-fDescent
				= seq [	drawAt { x=0, y=baseline} s1,
						drawAt { x=dw, y=baseline} (":"+++s2)
					  ] picture
	
		group_by :: !Int [x] -> [[x]];
		group_by n [] = [];
		group_by n l = [(take n l ) : (group_by n (drop n l))]; 
		  
